
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Sun Feb 20 12:09:37 2005
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief A brief description of this module 
   *
   *  \version CRYS_ATP_TST_DH_Acceptance.c#1:csrc:1
   *  \author doronp
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */



/***** Include Files ***********/ 
#include "CRYS.h"
 
#include "CRYS_RSA_Types.h" 
#include "CRYS_DH.h"

#include "CRYS_ATP_TST_DH_Acceptance_test_vectors.h"
#include "CRYS_ATP_TST_Defines.h"
#include "CRYS_ATP_TST_UTIL_Funcs.h"

#include "DX_VOS_Mem.h"
#if TST_USE_DH_DATA/*In case no test vectors are selected - code disabled*/
/******** Defines ********/ 
/******** Enums **********/ 
/******** Typedefs *********/ 
/******** Global Data ********/ 

/* modulus P value */
DxUint32_t   TST_DH_ModP[CRYS_DH_MAX_MOD_BUFFER_SIZE_IN_WORDS] = {0};

/* order Q value */
DxUint32_t   TST_DH_OrderQ[CRYS_DH_MAX_MOD_BUFFER_SIZE_IN_WORDS] = {0};

/* generator G value */
DxUint32_t   TST_DH_GeneratorG[CRYS_DH_MAX_MOD_BUFFER_SIZE_IN_WORDS] = {0};

/* integer factor J */
DxUint32_t   TST_DH_FactorJ[CRYS_DH_MAX_MOD_BUFFER_SIZE_IN_WORDS] = {0};

/* temp buffer for internal calculations */
CRYS_DHKGData_t        TST_DH_DHKGbuff = {0};

/* temp buffer for internal calculations */
CRYS_DHKG_CheckTemp_t  TST_TempBuff_ptr = {0};


/***** Private function prototype ******/ 
/******** Public Functions *****/ 
/***************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_DH_Acceptance_PKCS3        
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for the Diffie-Hellman key agreement 
 *  scheme according to PKCS#3  
 *  Prime sizes used (in bits) - 512/1024/2048                                  
 *  Generator sizes used (in bits) - 512/1024/2048
 *  L sizes used - 10/11/12 
 *
 * Algorithm:       
 * 1. Call the CRYS_DH_PKCS3_GeneratePubPrv function to 
 *    generate a public and privet key for the client
 * 2. Call the CRYS_DH_PKCS3_GeneratePubPrv function to 
 *    generate a public and privet key for the Server
 * 3. Call the CRYS_DH_PKCS3_GetSecretKey function to 
 *    generate a secret key for the client  
 * 4. Call the CRYS_DH_PKCS3_GetSecretKey function to 
 *    generate a secret key for the Server
 * 5. Compare the two secret keys - to verify they are identical                                                    
 *
 **************************/
CRYSError_t CRYS_ATP_TST_DH_Acceptance_PKCS3(void)
{
    int          TST_Index; 
    CRYSError_t  TST_Status=0;
    char         TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_DH_Acceptance_PKCS3";
    
    static CRYS_RSAUserPubKey_t TST_UserPubKey; 
    static CRYS_RSAPrimeData_t  TST_PrimeData;
    static DxUint8_t                  TST_ClientPrvKey[TST_MAX_DH_KEY_SIZE];
    static DxUint16_t                 TST_ClientPrvKeySize;
    static DxUint8_t                  TST_ClientPubKey[TST_MAX_DH_KEY_SIZE];
    static DxUint16_t                 TST_ClientPubKeySize;
    static DxUint8_t                  TST_ServerPrvKey[TST_MAX_DH_KEY_SIZE];
    static DxUint16_t                 TST_ServerPrvKeySize;
    static DxUint8_t                  TST_ServerPubKey[TST_MAX_DH_KEY_SIZE];
    static DxUint16_t                 TST_ServerPubKeySize;   
    static DxUint8_t                  TST_ClientSecretKey[TST_MAX_DH_KEY_SIZE];
    static DxUint16_t                 TST_ClientSecretKeySize;
    static DxUint8_t                  TST_ServerSecretKey[TST_MAX_DH_KEY_SIZE];
    static DxUint16_t                 TST_ServerSecretKeySize;        

    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************* \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_DH_Acceptance_PKCS3 Tests ******* \n"));        
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************* \n \n")); 

    /****************/
    /***** Start of test ********/
    /****************/  
    for (TST_Index = 0 ;TST_Index < TST_NumberOf_DH_PKCS3_AcceptanceTests ; TST_Index++)
    {       
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n")); 
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"   %s [test 1.%d] \n",TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_Name,TST_Index ));
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n")); 
    
        /* Generating Keys for the Client side */
        /*-------------------------------------*/   
        TST_ClientPrvKeySize = TST_ClientPubKeySize = TST_ServerPrvKeySize = TST_ServerPubKeySize =  TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_PrimeSize;

        TST_Status =  CRYS_DH_PKCS3_GeneratePubPrv(TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_Generator,
                                                      TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_GeneratorSize,
                                                      TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_Prime,
                                                      TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_PrimeSize,
                                                      TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_L,
                                                      &TST_UserPubKey,
                                                      &TST_PrimeData,
                                                      TST_ClientPrvKey,&TST_ClientPrvKeySize,
                                                      TST_ClientPubKey,&TST_ClientPubKeySize);

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                  "CRYS_DH_PKCS3_GeneratePubPrv - Client",
                                                  TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_Name,
                                                  TST_MasterTestName);


         ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                    "\n TST_ClientPrvKey Key--\n",
                                    TST_ClientPrvKey,
                                    TST_ClientPrvKeySize);   

         ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                    "\n TST_ClientPubKey Key--\n",
                                    TST_ClientPubKey,
                                    TST_ClientPubKeySize);
        
        /* Generating Keys for the Server side */
        /*-------------------------------------*/   
        TST_Status =  CRYS_DH_PKCS3_GeneratePubPrv(TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_Generator,
                                                  TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_GeneratorSize,
                                                  TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_Prime,
                                                  TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_PrimeSize,
                                                  TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_L,
                                                  &TST_UserPubKey,
                                                  &TST_PrimeData,
                                                  TST_ServerPrvKey,&TST_ServerPrvKeySize,
                                                  TST_ServerPubKey,&TST_ServerPubKeySize);

       TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                  "CRYS_DH_PKCS3_GeneratePubPrv - Server",
                                  TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_Name,
                                  TST_MasterTestName);
   

         ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                    "\n TST_ClientPrvKey Key--\n",
                                    TST_ServerPrvKey,
                                    TST_ServerPrvKeySize);   

         ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                    "\n TST_ClientPubKey Key--\n",
                                    TST_ServerPubKey,
                                    TST_ServerPubKeySize);

        /* Generating Secret Key for the client side */ 
        /*-------------------------------------------*/ 
        TST_ServerSecretKeySize = TST_ClientSecretKeySize = TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_PrimeSize;

        TST_Status =  CRYS_DH_PKCS3_GetSecretKey(TST_ClientPrvKey,
                                                    TST_ClientPrvKeySize,
                                                    TST_ServerPubKey,
                                                    TST_ServerPubKeySize,                                   
                                                    TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_Prime,
                                                    TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_PrimeSize,
                                                    &TST_UserPubKey,&TST_PrimeData,                                     
                                                    TST_ClientSecretKey,&TST_ClientSecretKeySize);      
        
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                              "CRYS_DH_PKCS3_GetSecretKey - Client",
                                              TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_Name,
                                              TST_MasterTestName);
  
        
        /* Generating Secret Key for the server side */ 
        /*-------------------------------------------*/ 
        TST_Status =  CRYS_DH_PKCS3_GetSecretKey(TST_ServerPrvKey,
                                                    TST_ServerPrvKeySize,
                                                    TST_ClientPubKey,
                                                    TST_ClientPubKeySize,                                   
                                                    TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_Prime,
                                                    TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_PrimeSize,
                                                    &TST_UserPubKey,&TST_PrimeData,                                     
                                                    TST_ServerSecretKey,&TST_ServerSecretKeySize);      
        
        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                              "CRYS_DH_PKCS3_GetSecretKey - Server",
                                              TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_Name,
                                              TST_MasterTestName);
 
        
        /* Verifying we have the same Secret Key both for client & Server */
        /*----------------------------------------------------------------*/                                                    
        TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_ServerSecretKey ,
                                                TST_ClientSecretKey,
                                                TST_ServerSecretKeySize);   
                                
        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                    "\nServer Secret Key--\n",
                                    TST_ServerSecretKey,
                                    TST_ServerSecretKeySize);   
                                    
        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                    "\nClient Secret Key--\n",
                                    TST_ClientSecretKey,
                                    TST_ClientSecretKeySize);                                                                                   

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                              "DX_VOS_MemCmp", 
                                              TST_DH_Acceptance_PKCS3_TestDataStruct[TST_Index].TST_Name,
                                              TST_MasterTestName);

                                        
    }/* End of TST_Index loop*/ 
    return TST_Status;
    
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}


/*****************************************************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*****************************************************/
/***************************
* Function Name:                                                           
*   CRYS_ATP_TST_DH_Acceptance_ANSI9_42     
* 
* Inputs:
*   None
*
* Outputs:
*   CRYSError_t - Function error return                        
*
* Description:                                  
*   This function Is a Acceptance test for the Diffie-Hellman key agreement 
*  schemes according to ANSI9.42 
*  using two methods for getting the secret data 
*  and using two methods for key derivation
*   Prime sizes used (in bits) - 1024/2048                  
*   Generator sizes used (in bits) - 1024/2048
*   Q sizes used (in bits) - 160
*  Derived Key sizes (in bits) - 128/256
*
* Algorithm:    
* 1. Call the CRYS_DH_ANSI_X942_GeneratePubPrv function to 
*    generate a public and privet key for the client
* 2. Call the CRYS_DH_ANSI_X942_GeneratePubPrv function to 
*    generate a public and privet key for the Server
* 3. Repete steps 1 & 2 in case of Hybrid scheam
* 4. Check the server and client public keys according to their domain parameters
* 5. Call the CRYS_DH_X942_GetSecret_ASN1_2_Data
*             or 
*          CRYS_DH_X942_GetSecret_Concat_2_Data
*             and in case of Hybrid scheam
*             CRYS_DH_X942_Hybrid_GetSecret_ASN1_2_Data
*             or 
*             CRYS_DH_X942_Hybrid_GetSecret_Concat_2_Data
*     both for the server and the Client to generate the secret keys
* 6. Compare the two secret keys - to verify they are identical                          
*
**************************/
CRYSError_t CRYS_ATP_TST_DH_Acceptance_ANSI942(void)
{
    int                  TST_Index,TST_KeyGetSecretKeyType,TST_SecretDataGenType;
    CRYSError_t          TST_Status=0;
    char                 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_DH_Acceptance_ANSI942";
#if TST_DH_DOMAIN_PARAMETERS_SUPPORTED
  CRYS_DHKG_ExpTemp_t  TST_TempBuff;
#endif
    
    static CRYS_RSAUserPubKey_t       TST_UserPubKey; 
    static CRYS_RSAPrimeData_t        TST_PrimeData;
  static DxUint8_t                  TST_ClientPrvKey1[TST_MAX_DH_KEY_SIZE]; 
    static DxUint16_t                 TST_ClientPrvKeySize1;
    static DxUint8_t                  TST_ClientPrvKey2[TST_MAX_DH_KEY_SIZE]; 
    static DxUint16_t                 TST_ClientPrvKeySize2;  
    static DxUint8_t                  TST_ClientPubKey1[TST_MAX_DH_KEY_SIZE]; 
    static DxUint16_t                 TST_ClientPubKeySize1;
    static DxUint8_t                  TST_ClientPubKey2[TST_MAX_DH_KEY_SIZE]; 
    static DxUint16_t                 TST_ClientPubKeySize2;  
    static DxUint8_t                  TST_ServerPrvKey1[TST_MAX_DH_KEY_SIZE]; 
    static DxUint16_t                 TST_ServerPrvKeySize1;
    static DxUint8_t                  TST_ServerPrvKey2[TST_MAX_DH_KEY_SIZE]; 
    static DxUint16_t                 TST_ServerPrvKeySize2;  
    static DxUint8_t                  TST_ServerPubKey1[TST_MAX_DH_KEY_SIZE]; 
    static DxUint16_t                 TST_ServerPubKeySize1;
    static DxUint8_t                  TST_ServerPubKey2[TST_MAX_DH_KEY_SIZE]; 
    static DxUint16_t                 TST_ServerPubKeySize2;  
    static DxUint8_t                  TST_ClientSecretKey[TST_MAX_DH_KEY_SIZE];
    static DxUint16_t                 TST_ClientSecretKeySize;
    static DxUint8_t                  TST_ServerSecretKey[TST_MAX_DH_KEY_SIZE];
    static DxUint16_t                 TST_ServerSecretKeySize;
    

    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************* \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_DH_Acceptance_ANSI942 Tests ******* \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************* \n \n")); 

  /****************/
    /***** Start of test ********/
    /****************/
    for (TST_SecretDataGenType = 0; TST_SecretDataGenType < TST_DH_SECRET_DATA_GEN_TYPES ;TST_SecretDataGenType++)
    { 
        for (TST_KeyGetSecretKeyType =0 ;TST_KeyGetSecretKeyType < TST_DH_ANSI924_DERIVATION_MODES ; TST_KeyGetSecretKeyType++) 
        {   
            for (TST_Index = 0 ;TST_Index < TST_NumberOf_DH_ANSI942_AcceptanceTests ; TST_Index++)
            {       
                /* according to standart ANSI942 doesn't work with prime size  = 512 bit */
                if (TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize == 64)
                    continue;
                
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n")); 
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  %s [test 2.%d] \n",TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name,TST_Index ));
                if (TST_SecretDataGenType == TST_DH_NON_HYBRID_MODE)
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  NON Hybrid secret data derivation \n"));                
                else
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  Hybrid secret data derivation \n"));
                if (TST_KeyGetSecretKeyType == TST_DH_GET_SECRET_KEY_ASN1)
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  ASN1 KeyDerivation SHA1 Func \n")); 
                else
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  Concat Key Derivation SHA1 Func \n"));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n")); 

      
                /* Generating Keys for the Client side */
                /*-------------------------------------*/  
                TST_ClientPrvKeySize1 = TST_ClientPubKeySize1 = 
                TST_ClientPrvKeySize2 = TST_ClientPubKeySize2 =
                TST_ServerPrvKeySize1 = TST_ServerPubKeySize1 =  
                TST_ServerPrvKeySize2 = TST_ServerPubKeySize2 = (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize;

                TST_Status = CRYS_DH_ANSI_X942_GeneratePubPrv(TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Generator, 
                                                                 (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_GeneratorSize, 
                                                                 TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime, 
                                                                 (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize, 
                                                                 TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Q, 
                                                                 (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_QSize, 
                                                                 &TST_UserPubKey,
                                                                 &TST_PrimeData,
                                                                 TST_ClientPrvKey1,&TST_ClientPrvKeySize1,
                                                                 TST_ClientPubKey1,&TST_ClientPubKeySize1); 

                TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                      "CRYS_DH_ANSI_X942_GeneratePubPrv - Client",
                                                      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
                                                      TST_MasterTestName);
  
                
                /* Generating Keys for the Server side */
                /*-------------------------------------*/   
                TST_Status = CRYS_DH_ANSI_X942_GeneratePubPrv(TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Generator, 
                                                             (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_GeneratorSize, 
                                                             TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime, 
                                                             (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize, 
                                                             TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Q, 
                                                             (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_QSize, 
                                                             &TST_UserPubKey,
                                                             &TST_PrimeData,
                                                             TST_ServerPrvKey1,&TST_ServerPrvKeySize1,
                                                             TST_ServerPubKey1,&TST_ServerPubKeySize1); 

                TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                      "CRYS_DH_ANSI_X942_GeneratePubPrv - Server",
                                                      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
                                                      TST_MasterTestName);
   

                /* Generating another Pair of Secret Keys both for Client and Serve */
                /* In case of working with Hybrid mode                              */
                /*------------------------------------------------------------------*/
                if (TST_SecretDataGenType == TST_DH_HYBRID_MODE)
                {
                    /* Generating Keys #2 for the Client side */
                    /*----------------------------------------*/    
                    TST_Status = CRYS_DH_ANSI_X942_GeneratePubPrv(TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Generator, 
                                                                 (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_GeneratorSize, 
                                                                 TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime, 
                                                                 (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize, 
                                                                 TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Q, 
                                                                 (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_QSize, 
                                                                 &TST_UserPubKey,
                                                                 &TST_PrimeData,
                                                                 TST_ClientPrvKey2,&TST_ClientPrvKeySize2,
                                                                 TST_ClientPubKey2,&TST_ClientPubKeySize2); 

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                          "CRYS_DH_ANSI_X942_GeneratePubPrv - Hybrid #2 - Client",
                                                          TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
                                                          TST_MasterTestName);
                     
                    /* Generating Keys #2 for the Server side */
                    /*----------------------------------------*/    
                    TST_Status = CRYS_DH_ANSI_X942_GeneratePubPrv(TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Generator, 
                                                                 (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_GeneratorSize, 
                                                                 TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime, 
                                                                 (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize, 
                                                                 TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Q, 
                                                                 (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_QSize, 
                                                                 &TST_UserPubKey,
                                                                 &TST_PrimeData,
                                                                 TST_ServerPrvKey2,&TST_ServerPrvKeySize2,
                                                                 TST_ServerPubKey2,&TST_ServerPubKeySize2); 

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                          "CRYS_DH_ANSI_X942_GeneratePubPrv - Hybrid #2 - Server",
                                                          TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
                                                          TST_MasterTestName);
                
                }

                TST_ClientSecretKeySize = TST_ServerSecretKeySize = TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_KeyLenInBits;

        #if TST_DH_DOMAIN_PARAMETERS_SUPPORTED
        /* check the server public key according to its domain parameters */
        /****************************************************************************/
        TST_Status = CRYS_DH_CheckPubKey( 
          TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime, 
          TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize,
          TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Q,
          TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_QSize,
          TST_ServerPubKey1,
          TST_ServerPubKeySize1,
          &TST_TempBuff); 

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
          "CRYS_DH_CheckPubKey - Server",
          TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
          TST_MasterTestName);

 
        
        /* check the client public key according to its domain parameters */
        /****************************************************************************/
        TST_Status = CRYS_DH_CheckPubKey( 
          TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime, 
          TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize,
          TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Q,
          TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_QSize,
          TST_ClientPubKey1,
          TST_ClientPubKeySize1,
          &TST_TempBuff); 

        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
          "CRYS_DH_CheckPubKey - Client",
          TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
          TST_MasterTestName);

 
        #endif
                if (TST_SecretDataGenType == TST_DH_NON_HYBRID_MODE)
                {
          if (TST_KeyGetSecretKeyType == TST_DH_GET_SECRET_KEY_ASN1)
                    {
            /* Generating Secret key for the Client */
                        /*--------------------------------------*/
                        TST_Status = CRYS_DH_X942_GetSecret_ASN1_2_Data(&TST_UserPubKey, 
                                                           &TST_PrimeData,
                                                           CRYS_DH_HASH_SHA1_mode,
                                                           TST_ClientPrvKey1,TST_ClientPrvKeySize1, 
                                                           TST_ServerPubKey1,TST_ServerPubKeySize1, 
                                                           TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime,
                                                           (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize,
                                                           TST_ClientSecretKey, 
                                                           TST_ClientSecretKeySize);
                        
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                              "CRYS_DH_X942_GetSecret_ASN1_2_Data - Client",
                                                              TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
                                                              TST_MasterTestName);

 

            /* Generating Secret key for the Server */
                        /*--------------------------------------*/
                        TST_Status = CRYS_DH_X942_GetSecret_ASN1_2_Data(&TST_UserPubKey, 
                                                           &TST_PrimeData,
                                                           CRYS_DH_HASH_SHA1_mode,
                                                           TST_ServerPrvKey1,TST_ServerPrvKeySize1, 
                                                           TST_ClientPubKey1,TST_ClientPubKeySize1, 
                                                           TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime,
                                                           (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize,
                                                           TST_ServerSecretKey, 
                                                           TST_ServerSecretKeySize);
                        
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                              "CRYS_DH_X942_GetSecret_ASN1_2_Data - Server",
                                                              TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
                                                              TST_MasterTestName);                                                            

  
                    }
                    else
                    {
                        /* Generating Secret key for the Client */
                        /*--------------------------------------*/                  
                        TST_Status = CRYS_DH_X942_GetSecret_Concat_2_Data(&TST_UserPubKey,
                                                             &TST_PrimeData,
                                                             CRYS_DH_HASH_SHA1_mode,
                                                             TST_ClientPrvKey1,TST_ClientPrvKeySize1,
                                                             TST_ServerPubKey1,TST_ServerPubKeySize1,
                                                             TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime, 
                                                             (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize, 
                                                             TST_ClientSecretKey,
                                                             TST_ClientSecretKeySize); 
                        
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                              "CRYS_DH_X942_GetSecret_Concat_2_Data - Client",
                                                              TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
                                                              TST_MasterTestName);


                        
                        /* Generating Secret key for the Server */
                        /*--------------------------------------*/                  
                        TST_Status = CRYS_DH_X942_GetSecret_Concat_2_Data(&TST_UserPubKey,
                                                             &TST_PrimeData,
                                                             CRYS_DH_HASH_SHA1_mode,
                                                             TST_ServerPrvKey1,TST_ServerPrvKeySize1,
                                                             TST_ClientPubKey1,TST_ClientPubKeySize1,
                                                             TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime, 
                                                             (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize, 
                                                             TST_ServerSecretKey,
                                                             TST_ServerSecretKeySize); 
                        
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                              "CRYS_DH_X942_GetSecret_Concat_2_Data - Server",
                                                              TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
                                                              TST_MasterTestName);  
 
                    }
                }
                else
                {
          #if TST_DH_DOMAIN_PARAMETERS_SUPPORTED
          /* check the server public key according to its domain parameters */
          /****************************************************************************/
          TST_Status = CRYS_DH_CheckPubKey( 
            TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime, 
            TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize,
            TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Q,
            TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_QSize,
            TST_ServerPubKey2,
            TST_ServerPubKeySize2,
            &TST_TempBuff); 

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
            "CRYS_DH_CheckPubKey - Server 2",
            TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
            TST_MasterTestName);

 

          /* check the client public key according to its domain parameters */
          /****************************************************************************/
          TST_Status = CRYS_DH_CheckPubKey( 
            TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime, 
            TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize,
            TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Q,
            TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_QSize,
            TST_ClientPubKey2,
            TST_ClientPubKeySize2,
            &TST_TempBuff); 

          TST_ATP_CHECK_ERROR_RETURN(TST_Status,
            "CRYS_DH_CheckPubKey - Client 2",
            TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
            TST_MasterTestName);

  
          #endif
          
          if (TST_KeyGetSecretKeyType == TST_DH_GET_SECRET_KEY_ASN1)
                    {
                        /* Generating Secret key for the Client */
                        /*--------------------------------------*/                      
                        TST_Status = CRYS_DH_X942_Hybrid_GetSecret_ASN1_2_Data(&TST_UserPubKey,
                                                                  &TST_PrimeData,CRYS_DH_HASH_SHA1_mode,
                                                                  TST_ClientPrvKey1,TST_ClientPrvKeySize1,
                                                                  TST_ClientPrvKey2,TST_ClientPrvKeySize2,
                                                                  TST_ServerPubKey1,TST_ServerPubKeySize1,
                                                                  TST_ServerPubKey2,TST_ServerPubKeySize2,
                                                                  TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime,
                                                                  (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize,
                                                                  TST_ClientSecretKey,
                                                                  TST_ClientSecretKeySize);

                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                              "CRYS_DH_X942_Hybrid_GetSecret_ASN1_2_Data - Client", 
                                                              TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
                                                              TST_MasterTestName);  

  
                        
                        /* Generating Secret key for the Server */
                        /*--------------------------------------*/                      
                        TST_Status = CRYS_DH_X942_Hybrid_GetSecret_ASN1_2_Data(&TST_UserPubKey,
                                                                  &TST_PrimeData,CRYS_DH_HASH_SHA1_mode,
                                                                  TST_ServerPrvKey1,TST_ServerPrvKeySize1,
                                                                  TST_ServerPrvKey2,TST_ServerPrvKeySize2,
                                                                  TST_ClientPubKey1,TST_ClientPubKeySize1,
                                                                  TST_ClientPubKey2,TST_ClientPubKeySize2,
                                                                  TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime,
                                                                  (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize,
                                                                  TST_ServerSecretKey,
                                                                  TST_ServerSecretKeySize);

                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                              "CRYS_DH_X942_Hybrid_GetSecret_ASN1_2_Data - Server", 
                                                              TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
                                                              TST_MasterTestName);                                                                                                                        

  
                    }
                    else
                    {
                        /* Generating Secret key for the Client */
                        /*--------------------------------------*/                      
                        TST_Status = CRYS_DH_X942_Hybrid_GetSecret_Concat_2_Data(&TST_UserPubKey,
                                                                    &TST_PrimeData,CRYS_DH_HASH_SHA1_mode,
                                                                    TST_ClientPrvKey1,TST_ClientPrvKeySize1,
                                                                    TST_ClientPrvKey2,TST_ClientPrvKeySize2,
                                                                    TST_ServerPubKey1,TST_ServerPubKeySize1,
                                                                    TST_ServerPubKey2,TST_ServerPubKeySize2,
                                                                    TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime,
                                                                    (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize,
                                                                    TST_ClientSecretKey,TST_ClientSecretKeySize);

                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                              "CRYS_DH_X942_Hybrid_GetSecret_Concat_2_Data - Client",
                                                              TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
                                                              TST_MasterTestName);  

 

                        /* Generating Secret key for the Server */
                        /*--------------------------------------*/                      
                        TST_Status = CRYS_DH_X942_Hybrid_GetSecret_Concat_2_Data(&TST_UserPubKey,
                                                                    &TST_PrimeData, 
                                                                    CRYS_DH_HASH_SHA1_mode, 
                                                                    TST_ServerPrvKey1,TST_ServerPrvKeySize1,
                                                                    TST_ServerPrvKey2,TST_ServerPrvKeySize2,
                                                                    TST_ClientPubKey1,TST_ClientPubKeySize1,
                                                                    TST_ClientPubKey2,TST_ClientPubKeySize2,
                                                                    TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime,
                                                                    (DxUint16_t)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize,
                                                                    TST_ServerSecretKey,TST_ServerSecretKeySize);

                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                              "CRYS_DH_X942_Hybrid_GetSecret_Concat_2_Data - Server",
                                                              TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
                                                              TST_MasterTestName);                                                                                  

 
                    }               
                }

                /* Verifying we have the same Secret Key both for client & Server */
                /*----------------------------------------------------------------*/                                                    
                TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_ServerSecretKey ,
                                                     TST_ClientSecretKey,
                                                    (TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_KeyLenInBits)/8);  
                                        
                ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                            "\nServer Secret Key--\n",
                                            TST_ServerSecretKey,
                                            (TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_KeyLenInBits)/8);  
                                            
                ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                            "\nClient Secret Key--\n",
                                            TST_ClientSecretKey,
                                            (TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_KeyLenInBits)/8);

                TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                                      "DX_VOS_MemCmp", 
                                                      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
                                                      TST_MasterTestName);
 
            }/*End of TST_Index loop*/                                  
        }/*End of TST_KeyDerivationMode loop */ 
    }/*End of TST_SecretDataGenType loop*/

    return TST_Status;                          

EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}
/*****************************************************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*****************************************************/
/***************************
* Function Name:                                                           
*  CRYS_ATP_TST_DH_DomainParametersGeneration_Acceptance_ANSI942         
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for the Diffie-Hellman domain parameters
*  generation according to ANSI9.42 
*  Prime sizes used (in bits) - 1024/2048                                  
*  Generator sizes used (in bits) - 1024/2048
*  Q sizes used (in bits) - 160
*  Derived Key sizes (in bits) - 128/256
*
* Algorithm:       
* 1. Call the CRYS_DH_CreateDomainParams function to 
*    generate (p, q, g)
* 2. Compare (p, q, g) with expected (p, q, g)
* 3. Call the CRYS_DH_CheckDomainParams function - to verify that function passes
*
**************************/
#if TST_DH_DOMAIN_PARAMETERS_SUPPORTED
CRYSError_t CRYS_ATP_TST_DH_DomainParametersGeneration_Acceptance_ANSI942(void)
{
  int                  TST_Index;
  CRYSError_t          TST_Status=0;
  char                 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_DH_DomainParametersGeneration_Acceptance_ANSI942";
  /* pointer to modulus (prime) P */
  DxUint8_t       *TST_ModP      = (DxUint8_t*)TST_DH_ModP;          
  /* pointer to order Q of the generator */
  DxUint8_t            *TST_Q         = (DxUint8_t*)TST_DH_OrderQ; 
  /* pointer to the generator */
  DxUint8_t            *TST_Generator = (DxUint8_t*)TST_DH_GeneratorG;                
  /* pointer to the integer factor J */
  DxUint8_t            *TST_factorJ   = (DxUint8_t*)TST_DH_FactorJ;  
  /* counter of tries to generate the prime */
  DxUint32_t            TST_PgenCounter;
  DxUint32_t            TST_GeneratorSizeBytes = CRYS_DH_MAX_MOD_SIZE_IN_BYTES;
  DxUint32_t            TST_JsizeBytes; 
  DxInt8_t              generateSeed = 0; // seed is given as input

  ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"****************************************************************************** \n"));
  ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_DH_DomainParametersGeneration_Acceptance_ANSI942 Tests ******* \n"));      
  ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"****************************************************************************** \n \n")); 

  /****************************/
  /***** Start of test ********/
  /****************************/
  for (TST_Index = 0 ;TST_Index < TST_NumberOf_DH_ANSI942_AcceptanceTests ; TST_Index++)
  {       
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n")); 
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"  %s [test 3.%d] \n",TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name,TST_Index ));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n")); 
    
    TST_GeneratorSizeBytes = CRYS_DH_MAX_MOD_SIZE_IN_BYTES;
    TST_JsizeBytes         = CRYS_DH_MAX_MOD_SIZE_IN_BYTES;      

    /* Only supported TST_PrimeSize bigger than 1024Bits */
    if (TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize*8 < CRYS_DH_MIN_VALID_KEY_SIZE_VALUE_IN_BITS)
    {
      continue;
    }
    /* create domain parameters */
    /****************************/
    TST_Status = CRYS_DH_CreateDomainParams( 
      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize * 8,
      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_QSize * 8,
      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_SeedSize,
      TST_ModP,
      TST_Q,
      TST_Generator,
      &TST_GeneratorSizeBytes,
      TST_factorJ,
      &TST_JsizeBytes,
      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Seed,
      generateSeed,            
      &TST_PgenCounter,
      &TST_DH_DHKGbuff);

    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_DH_CreateDomainParams is done\n"));
      
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
      "CRYS_DH_CreateDomainParams",
      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
      TST_MasterTestName);




    if( generateSeed != 1 )
    {
      /* Verifying we have expected modulus P value */
      /*----------------------------------------------------------------*/                                                    
      TST_Status = (CRYSError_t)DX_VOS_MemCmp(
                     TST_ModP,
                                 TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime,
                     TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize);  

      /*Print the outputted and then the expected result after mem compare */
      ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
        "\nData outputted by engine [output buffer]-- \n",
        (DxUint8_t*)TST_ModP, 
        TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize); 

      ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
        "\nData Expected by engine --\n",
        (DxUint8_t*)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Prime,
        TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize);                              

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
        "CRYS_DH_CreateDomainParams - verify modulus P",
        TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
        TST_MasterTestName);

  

      /* Verifying we have expected order Q value */
      /*------------------------------------------*/                                                    
      TST_Status = (CRYSError_t)DX_VOS_MemCmp(
        TST_Q,
        TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Q,
        TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_QSize);  

      /*Print the outputted and then the expected result after mem compare */
      ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
        "\nData outputted by engine [output buffer]-- \n",
        (DxUint8_t*)TST_Q, 
        TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_QSize); 

      ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
        "\nData Expected by engine --\n",
        (DxUint8_t*)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Q,
        TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_QSize);                              

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
        "CRYS_DH_CreateDomainParams - verify order Q",
        TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
        TST_MasterTestName);



      /* Verifying we have expected factorJ value */
      /*------------------------------------------*/                                                    
      TST_Status = (CRYSError_t)DX_VOS_MemCmp(
        TST_factorJ,
        TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_FactorJ,
        TST_JsizeBytes);  

      /*Print the outputted and then the expected result after mem compare */
      ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
        "\nData outputted by engine [output buffer]-- \n",
        (DxUint8_t*)TST_factorJ, 
        TST_JsizeBytes); 

      ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
        "\nData Expected by engine --\n",
        (DxUint8_t*)TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_FactorJ,
        TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_FactorJSize);                              

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
        "CRYS_DH_CreateDomainParams - verify factor J",
        TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
        TST_MasterTestName);



      
    }

    /* Check that function, that checks domain parameters, returns success */
    /*********************************************************************/
    TST_Status = CRYS_DH_CheckDomainParams( 
      TST_ModP,  
      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_PrimeSize, 
      TST_Q,         /*in */
      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_QSize,    /*in */
      TST_Generator,     /*in */
      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_GeneratorSize, /*in */
      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Seed,       
      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_SeedSize,
      TST_PgenCounter,
      &TST_TempBuff_ptr,
      &TST_DH_DHKGbuff);

    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_DH_CheckDomainParams is done\n"));

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
      "CRYS_DH_CheckDomainParams",
      TST_DH_Acceptance_ANSI942_TestDataStruct[TST_Index].TST_Name, 
      TST_MasterTestName);

    
  }/*End of TST_Index loop*/                                  
    
  return TST_Status;      

EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}

#endif

#endif



